#ifndef FASTRANDOM_HPP
#define FASTRANDOM_HPP

#include "Common.hpp"
#include <type_traits>

// fastRandom
//
// Generate a random number from 0 to range-1 without using division.
// Use as a faster alternative to std::uniform_int_distribution.
// Assumes that Gen returns a full 32 or 64 bits of random data.

template<typename Gen, typename = std::enable_if_t<std::is_same_v<typename Gen::result_type, uint64_t>>>
uint64_t fastRandom(uint64_t range, Gen& gen)
{
#if ARCH_64
    return ((__uint128_t)range * gen()) >> 64;
#else
    return gen() % range;
#endif
}

template<typename Gen, typename = std::enable_if_t<std::is_same_v<typename Gen::result_type, uint32_t>>>
uint32_t fastRandom(uint32_t range, Gen& gen)
{
    return ((uint64_t)range * gen()) >> 32;
}

// fastShuffle
//
// Shuffle the given range into a random order without using division.
// Use as a faster alternative to std::shuffle.
// Assumes that Gen returns a full 32 or 64 bits of random data.

template<class It, class Gen>
void fastShuffle(It first, It last, Gen& gen)
{
    size_t size = last - first;
    while (size > 1)
    {
        size_t pos = fastRandom(size, gen);
        size--;
        using std::swap;
        swap(*(first + pos), *(first + size));
    }
}

// fastRandomReal<T>
// fastRandomDouble
// fastRandomFloat
//
// 1 argument version: Generate a random real number between 0 and 1.
// 2 argument version: Generate a random real number between 0 and range.
// 3 argument version: Generate a random real number between low and high.
// Use as a faster alternative to std::uniform_real_distribution.
// Assumes that Gen returns a full 32 or 64 bits of random data.

template<typename RealType, typename Gen>
RealType fastRandomReal(Gen& gen)
{
    return (RealType)gen() * ((RealType)1 / (RealType)(typename Gen::result_type)-1);
}

template<typename RealType, typename Gen>
RealType fastRandomReal(RealType range, Gen& gen)
{
    return fastRandomReal<RealType>(gen) * range;
}

template<typename RealType, typename Gen>
RealType fastRandomReal(RealType low, RealType high, Gen& gen)
{
    return fastRandomReal<RealType>(gen) * (high - low) + low;
}

template<typename Gen>
double fastRandomDouble(Gen& gen)
{ return fastRandomReal<double>(gen); }

template<typename Gen>
double fastRandomDouble(double range, Gen& gen)
{ return fastRandomReal<double>(range, gen); }

template<typename Gen>
double fastRandomDouble(double low, double high, Gen& gen)
{ return fastRandomReal<double>(low, high, gen); }

template<typename Gen>
float fastRandomFloat(Gen& gen)
{ return fastRandomReal<float>(gen); }

template<typename Gen>
float fastRandomFloat(float range, Gen& gen)
{ return fastRandomReal<float>(range, gen); }

template<typename Gen>
float fastRandomFloat(float low, float high, Gen& gen)
{ return fastRandomReal<float>(low, high, gen); }

#endif
