#include "Http.hpp"
#include "Cleanup.hpp"
#include "StringUtils.hpp"
#include <curl/curl.h>

// sudo apt-get install libcurl4-openssl-dev

using std::string;
using std::vector;
using std::pair;

namespace Http
{
    void init()
    {
        CURLcode res = curl_global_init(CURL_GLOBAL_ALL);
        if (res != CURLE_OK)
        {
            fprintf(stderr, "curl_global_init failed\n");
            exit(1);
        }
    }

    void cleanup()
    {
        curl_global_cleanup();
    }

    size_t writeFunction(char* ptr, size_t size, size_t nmemb, void* userdata)
    {
        string* pStr = (string*)userdata;
        pStr->append(ptr, size * nmemb);
        return size * nmemb;
    }

    bool makeRequest(const string& url,
                     bool post,
                     const vector<pair<string, string>>* pQuery,
                     const vector<MimeEntry>* pMimeQuery,
                     const string* pRequest,
                     const string* pContentType,
                     const vector<string>* pHeaders,
                     string* pResponse,
                     string* pMsg)
    {
        CURL* curl = curl_easy_init();
        if (!curl)
        {
            if (pMsg) *pMsg = "curl_easy_init failed";
            return false;
        }
        Cleanup cleanupCurl([&](){ curl_easy_cleanup(curl); });

        string queryStr;
        if (pQuery)
        {
            bool first = true;
            for (auto& queryItem : *pQuery)
            {
                if (first)
                {
                    first = false;
                }
                else
                {
                    queryStr += "&";
                }

                {
                    char* key = curl_easy_escape(curl, queryItem.first.c_str(), queryItem.first.size());
                    if (!key)
                    {
                        if (pMsg) *pMsg = "curl_easy_escape failed";
                        return false;
                    }
                    queryStr += key;
                    curl_free(key);
                }

                queryStr += "=";

                {
                    char* value = curl_easy_escape(curl, queryItem.second.c_str(), queryItem.second.size());
                    if (!value)
                    {
                        if (pMsg) *pMsg = "curl_easy_escape failed";
                        return false;
                    }
                    queryStr += value;
                    curl_free(value);
                }
            }
        }

        curl_mime* mime = NULL;
        Cleanup cleanupMime([&](){ if (mime) curl_mime_free(mime); });
        if (pMimeQuery)
        {
            mime = curl_mime_init(curl);
            if (!mime)
            {
                if (pMsg) *pMsg = "curl_mime_init failed";
                return false;
            }

            for (auto& mimeEntry : *pMimeQuery)
            {
                curl_mimepart* mimepart = curl_mime_addpart(mime);
                if (!mimepart)
                {
                    if (pMsg) *pMsg = "curl_mime_addpart failed";
                    return false;
                }

                curl_mime_name(mimepart, mimeEntry.name.c_str());

                if (curl_mime_data(mimepart, mimeEntry.data.data(), mimeEntry.data.size()) != CURLE_OK)
                {
                    if (pMsg) *pMsg = "curl_mime_data failed";
                    return false;
                }

                if (!mimeEntry.fileName.empty())
                {
                    if (curl_mime_filename(mimepart, mimeEntry.fileName.c_str()) != CURLE_OK)
                    {
                        if (pMsg) *pMsg = "curl_mime_filename failed";
                        return false;
                    }
                }
            }
        }

        size_t contentLength = 0;
        const char* pContent = nullptr;
        if (post)
        {
            if (pRequest && pQuery)
            {
                if (pMsg) *pMsg = "Both request and query provided";
                return false;
            }
            if (pRequest && pMimeQuery)
            {
                if (pMsg) *pMsg = "Both request and mime query provided";
                return false;
            }
            if (pQuery && pMimeQuery)
            {
                if (pMsg) *pMsg = "Both query and mime query provided";
                return false;
            }
            if (pRequest)
            {
                contentLength = pRequest->size();
                pContent = pRequest->c_str();
            }
            else if (pQuery)
            {
                contentLength = queryStr.size();
                pContent = queryStr.c_str();
            }
            else
            {
                contentLength = 0;
                pContent = nullptr;
            }
        }

        struct curl_slist *headers = nullptr;
        Cleanup cleanupHeaders([&](){ curl_slist_free_all(headers); });
        if (post)
        {
            if (pContentType)
            {
                string contentTypeHeader = "Content-Type: " + *pContentType;
                headers = curl_slist_append(headers, contentTypeHeader.c_str());
            }

            if (!pMimeQuery)
            {
                string contentLengthHeader = strprintf("Content-Length: %" PRIuZ "", contentLength);
                headers = curl_slist_append(headers, contentLengthHeader.c_str());
            }
        }

        if (pHeaders)
        {
            for (auto& header : *pHeaders)
            {
                headers = curl_slist_append(headers, header.c_str());
            }
        }

        string strResponse;
        char errorBuf[CURL_ERROR_SIZE];

        string fullUrl;
        if (!post && !queryStr.empty())
        {
            fullUrl = url + "?" + queryStr;
        }
        else
        {
            fullUrl = url;
        }
        curl_easy_setopt(curl, CURLOPT_URL, fullUrl.c_str());
        //fprintf(stderr, "fullUrl: '%s'\n", fullUrl.c_str());

        //curl_easy_setopt(curl, CURLOPT_VERBOSE, (long)1);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, writeFunction);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void*)&strResponse);
        if (headers)
        {
            curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
        }
        curl_easy_setopt(curl, CURLOPT_PROTOCOLS_STR, "http,https");
#if PLATFORM_WINDOWS
        curl_easy_setopt(curl, CURLOPT_SSL_OPTIONS, (long)CURLSSLOPT_NATIVE_CA);
#endif
        curl_easy_setopt(curl, CURLOPT_ACCEPT_ENCODING, ""); // all supported
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1L);
        curl_easy_setopt(curl, CURLOPT_MAXREDIRS, 10L);
        curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, errorBuf);
        if (post)
        {
            curl_easy_setopt(curl, CURLOPT_POST, 1L);
            if (pMimeQuery)
            {
                curl_easy_setopt(curl, CURLOPT_MIMEPOST, mime);
            }
            else
            {
                curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE_LARGE, (curl_off_t)contentLength);
                if (contentLength != 0)
                {
                    curl_easy_setopt(curl, CURLOPT_POSTFIELDS, pContent);
                }
            }
        }

        errorBuf[0] = 0;

        CURLcode res = curl_easy_perform(curl);
        if (res != CURLE_OK)
        {
            if (pMsg)
            {
                *pMsg = "curl request failed";
                size_t errorLen = strlen(errorBuf);
                if (errorLen > 0)
                {
                    if (errorBuf[errorLen - 1] == '\n')
                    {
                        errorBuf[errorLen - 1] = 0;
                        errorLen--;
                    }
                }
                if (errorLen > 0)
                {
                    *pMsg += ": ";
                    *pMsg += errorBuf;
                }
            }
            return false;
        }

        long code = 0;
        res = curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &code);
        if (res != CURLE_OK)
        {
            if (pMsg) *pMsg = "curl_easy_getinfo(CURLINFO_RESPONSE_CODE) failed";
            return false;
        }

        //printf("%s\n", strResponse.c_str());

        if (code < 200 || code > 299)
        {
            //printf("%s\n", strResponse.c_str());
            if (pMsg) *pMsg = strprintf("curl response code %lu", code);
            return false;
        }

        if (pResponse)
        {
            *pResponse = std::move(strResponse);
        }

        return true;
    }

    bool get(const string& url,
             const vector<string>* pHeaders,
             string* pResponse,
             string* pMsg)
    {
        return makeRequest(url,
                           false,
                           nullptr,
                           nullptr,
                           nullptr,
                           nullptr,
                           pHeaders,
                           pResponse,
                           pMsg);
    }

    bool getQuery(const string& url,
                  const vector<pair<string, string>>& query,
                  const vector<string>* pHeaders,
                  string* pResponse,
                  string* pMsg)
    {
        return makeRequest(url,
                           false,
                           &query,
                           nullptr,
                           nullptr,
                           nullptr,
                           pHeaders,
                           pResponse,
                           pMsg);
    }

    bool post(const string& url,
              const string& request,
              const string& contentType,
              const vector<string>* pHeaders,
              string* pResponse,
              string* pMsg)
    {
        return makeRequest(url,
                           true,
                           nullptr,
                           nullptr,
                           &request,
                           &contentType,
                           pHeaders,
                           pResponse,
                           pMsg);
    }

    bool postQuery(const string& url,
                   const vector<pair<string, string>>& query,
                   const vector<string>* pHeaders,
                   string* pResponse,
                   string* pMsg)
    {
        string contentType("application/x-www-form-urlencoded");
        return makeRequest(url,
                           true,
                           &query,
                           nullptr,
                           nullptr,
                           &contentType,
                           pHeaders,
                           pResponse,
                           pMsg);
    }

    bool postMime(const string& url,
                  const vector<MimeEntry>& mimeQuery,
                  const vector<string>* pHeaders,
                  string* pResponse,
                  string* pMsg)
    {
        return makeRequest(url,
                           true,
                           nullptr,
                           &mimeQuery,
                           nullptr,
                           nullptr,
                           pHeaders,
                           pResponse,
                           pMsg);
    }
}
