#ifndef JSONUTILS_HPP
#define JSONUTILS_HPP

#include "Common.hpp"
#include "Json.hpp"

class JException
{
public:
    JException(const char* msg) noexcept : m_msg(msg) {}

    const char* m_msg;
};

[[noreturn]] inline void JError(const char* msg)
{ throw JException(msg); }

inline void JCheck(bool cond, const char* msg)
{ if (!cond) JError(msg); }

inline bool JGetBool_(const rapidjson::Value& jValue, const char* msg)
{ JCheck(jValue.IsBool(), msg); return jValue.GetBool(); }
inline int32_t JGetInt_(const rapidjson::Value& jValue, const char* msg)
{ JCheck(jValue.IsInt(), msg); return jValue.GetInt(); }
inline uint32_t JGetUint_(const rapidjson::Value& jValue, const char* msg)
{ JCheck(jValue.IsUint(), msg); return jValue.GetUint(); }
inline int64_t JGetInt64_(const rapidjson::Value& jValue, const char* msg)
{ JCheck(jValue.IsInt64(), msg); return jValue.GetInt64(); }
inline uint64_t JGetUint64_(const rapidjson::Value& jValue, const char* msg)
{ JCheck(jValue.IsUint64(), msg); return jValue.GetUint64(); }
inline double JGetDouble_(const rapidjson::Value& jValue, const char* msg)
{ JCheck(jValue.IsNumber(), msg); return jValue.GetDouble(); }
inline const char* JGetString_(const rapidjson::Value& jValue, const char* msg)
{ JCheck(jValue.IsString(), msg); return jValue.GetString(); }
inline const rapidjson::Value& JCheckObject_(const rapidjson::Value& jValue,
                                             const char* msg)
{ JCheck(jValue.IsObject(), msg); return jValue; }
inline const rapidjson::Value& JCheckArray_(const rapidjson::Value& jValue,
                                            const char* msg)
{ JCheck(jValue.IsArray(), msg); return jValue; }

#define JGetBool(jValue) JGetBool_(jValue, #jValue ": not bool")
#define JGetInt(jValue) JGetInt_(jValue, #jValue ": not int")
#define JGetUint(jValue) JGetUint_(jValue, #jValue ": not uint")
#define JGetInt64(jValue) JGetInt64_(jValue, #jValue ": not int64")
#define JGetUint64(jValue) JGetUint64_(jValue, #jValue ": not uint64")
#define JGetDouble(jValue) JGetDouble_(jValue, #jValue ": not double")
#define JGetString(jValue) JGetString_(jValue, #jValue ": not string")
#define JCheckObject(jValue) JCheckObject_(jValue, #jValue ": not object")
#define JCheckArray(jValue) JCheckArray_(jValue, #jValue ": not array")

inline const rapidjson::Document& JCheckParse_(const rapidjson::Document& jDocument,
                                               const char* msg)
{ JCheck(!jDocument.HasParseError(), msg); return jDocument; }

#define JCheckParse(jDocument) \
    JCheckParse_(jDocument, #jDocument ": parse failed")

#define JCheckParseObject(jDocument) \
    JCheckObject_(JCheckParse(jDocument), #jDocument ": not object")
#define JCheckParseArray(jDocument) \
    JCheckArray_(JCheckParse(jDocument), #jDocument ": not array")

inline const rapidjson::Value& JObjectGet_(const rapidjson::Value& jObject,
                                           const char* key,
                                           const char* msg)
{
    auto jit = jObject.FindMember(key);
    JCheck(jit != jObject.MemberEnd(), msg);
    return jit->value;
}

#define JObjectGet(jObject, key) \
    JObjectGet_(JCheckObject(jObject), key, #jObject ": '" key "' not found")

#define JObjectGetBool(jObject, key) \
    JGetBool_(JObjectGet(jObject, key), #jObject ": '" key "' not bool")
#define JObjectGetInt(jObject, key) \
    JGetInt_(JObjectGet(jObject, key), #jObject ": '" key "' not int")
#define JObjectGetUint(jObject, key) \
    JGetUint_(JObjectGet(jObject, key), #jObject ": '" key "' not uint")
#define JObjectGetInt64(jObject, key) \
    JGetInt64_(JObjectGet(jObject, key), #jObject ": '" key "' not int64")
#define JObjectGetUint64(jObject, key) \
    JGetUint64_(JObjectGet(jObject, key), #jObject ": '" key "' not uint64")
#define JObjectGetDouble(jObject, key) \
    JGetDouble_(JObjectGet(jObject, key), #jObject ": '" key "' not double")
#define JObjectGetString(jObject, key) \
    JGetString_(JObjectGet(jObject, key), #jObject ": '" key "' not string")
#define JObjectGetObject(jObject, key) \
    JCheckObject_(JObjectGet(jObject, key), #jObject ": '" key "' not object")
#define JObjectGetArray(jObject, key) \
    JCheckArray_(JObjectGet(jObject, key), #jObject ": '" key "' not array")

inline const rapidjson::Value& JArrayCheckSize_(const rapidjson::Value& jArray,
                                                rapidjson::SizeType size,
                                                const char* msg)
{ JCheck(jArray.Size() == size, msg); return jArray; }

#define JArrayCheckSize(jArray, size) \
    JArrayCheckSize_(JCheckArray(jArray), size, #jArray ": incorrect size");

inline const rapidjson::Value& JArrayGet_(const rapidjson::Value& jArray,
                                          rapidjson::SizeType idx,
                                          const char* msg)
{ JCheck(idx < jArray.Size(), msg); return jArray[idx]; }

#define JArrayGet(jArray, idx) \
    JArrayGet_(JCheckArray(jArray), idx, #jArray ": index '" #idx "' out of range")

#define JArrayGetBool(jArray, idx) \
    JGetBool_(JArrayGet(jArray, idx), #jArray ": index '" #idx "' not bool")
#define JArrayGetInt(jArray, idx) \
    JGetInt_(JArrayGet(jArray, idx), #jArray ": index '" #idx "' not int")
#define JArrayGetUint(jArray, idx) \
    JGetUint_(JArrayGet(jArray, idx), #jArray ": index '" #idx "' not uint")
#define JArrayGetInt64(jArray, idx) \
    JGetInt64_(JArrayGet(jArray, idx), #jArray ": index '" #idx "' not int64")
#define JArrayGetUint64(jArray, idx) \
    JGetUint64_(JArrayGet(jArray, idx), #jArray ": index '" #idx "' not uint64")
#define JArrayGetDouble(jArray, idx) \
    JGetDouble_(JArrayGet(jArray, idx), #jArray ": index '" #idx "' not double")
#define JArrayGetString(jArray, idx) \
    JGetString_(JArrayGet(jArray, idx), #jArray ": index '" #idx "' not string")
#define JArrayGetObject(jArray, idx) \
    JCheckObject_(JArrayGet(jArray, idx), #jArray ": index '" #idx "' not object")
#define JArrayGetArray(jArray, idx) \
    JCheckArray_(JArrayGet(jArray, idx), #jArray ": index '" #idx "' not array")

// JForEach(auto& jItem, jItems,
//          auto& item, items)
// {
//     parseItem(jItem, item);
// }

#define JForEach(decl_jItem, jItems, decl_item, items) \
    for (decl_jItem : JCheckArray(jItems), (items).clear(), (items).reserve(jItems.Size()), (jItems).GetArray()) \
        if (decl_item = (items).emplace_back(); false) ; else

#endif
