#include "ParseUtils.hpp"
#include <limits>
#include <type_traits>

using std::string;

namespace
{
    template<typename T>
    bool parseIntImpl(const string& str, T* pValue)
    {
        T base = 10;
        size_t size = str.size();
        if (size == 0) return false;

        bool neg = false;
        size_t pos = 0;
        if (std::is_signed<T>::value && str[0] == '-')
        {
            if (size == 1) return false;
            neg = true;
            pos = 1;
        }

        T value = 0;
        for (; pos < size; pos++)
        {
            if (neg)
            {
                if (value < std::numeric_limits<T>::min() / base)
                {
                    return false;
                }
            }
            else
            {
                if (value > std::numeric_limits<T>::max() / base)
                {
                    return false;
                }
            }
            value *= base;
            char ch = str[pos];
            T digit;
            if (ch >= '0' && ch <= '9')
            {
                digit = ch - '0';
            }
            else if (ch >= 'a' && ch <= 'z')
            {
                digit = ch - 'a' + 10;
            }
            else if (ch >= 'A' && ch <= 'Z')
            {
                digit = ch - 'A' + 10;
            }
            else
            {
                return false;
            }
            if (digit >= base)
            {
                return false;
            }
            if (neg)
            {
                if (-digit < std::numeric_limits<T>::min() - value)
                {
                    return false;
                }
                value -= digit;
            }
            else
            {
                if (digit > std::numeric_limits<T>::max() - value)
                {
                    return false;
                }
                value += digit;
            }
        }
        if (pValue)
        {
            *pValue = value;
        }
        return true;
    }

    template<typename T, typename F>
    bool parseFloatImpl(const string& str, T* pValue, F&& f)
    {
        {
            size_t size = str.size();
            size_t pos = 0;
            if (size != 0 && str[0] == '-')
            {
                pos = 1;
            }
            bool gotDot = false;
            bool gotDigit = false;
            for (; pos < size; pos++)
            {
                char ch = str[pos];
                if (ch == '.')
                {
                    if (gotDot) return false;
                    gotDot = true;
                }
                else if (ch >= '0' && ch <= '9')
                {
                    gotDigit = true;
                }
                else
                {
                    return false;
                }
            }
            if (!gotDigit)
            {
                return false;
            }
        }
        const char* nptr = str.c_str();
        char* endptr = nullptr;
        errno = 0;
        auto value = f(nptr, &endptr);
        if (errno != 0 || *nptr == '\0' || *endptr != '\0')
        {
            return false;
        }
        if (pValue)
        {
            *pValue = (T)value;
        }
        return true;
    }
}

bool parseU32(const string& str, uint32_t* pValue)
{
    return parseIntImpl(str, pValue);
}

bool parseI32(const string& str, int32_t* pValue)
{
    return parseIntImpl(str, pValue);
}

bool parseU64(const string& str, uint64_t* pValue)
{
    return parseIntImpl(str, pValue);
}

bool parseI64(const string& str, int64_t* pValue)
{
    return parseIntImpl(str, pValue);
}

bool parseFloat(const string& str, float* pValue)
{
    return parseFloatImpl(str, pValue, strtof);
}

bool parseDouble(const string& str, double* pValue)
{
    return parseFloatImpl(str, pValue, strtod);
}

namespace
{
    bool parseDurationImpl(const string& str, bool wantMS, uint64_t* pDuration)
    {
        bool gotH = false;
        bool gotM = false;
        bool gotS = false;
        string strH;
        string strM;
        string strS;
        size_t size = str.size();
        size_t pos = 0;
        size_t start = pos;
        bool gotDot = false;
        while (pos < size)
        {
            char ch = str[pos++];
            if (ch >= '0' && ch <= '9')
            {
            }
            else if (ch == '.')
            {
                if (!wantMS || gotDot) return false;
                gotDot = true;
            }
            else if (ch == 'h')
            {
                if (gotDot || gotH || gotM || gotS) return false;
                if (pos - 1 == start) return false;
                gotH = true;
                strH = str.substr(start, pos - 1 - start);
                start = pos;
                gotDot = false;
            }
            else if (ch == 'm')
            {
                if (gotDot || gotM || gotS) return false;
                if (pos - 1 == start) return false;
                gotM = true;
                strM = str.substr(start, pos - 1 - start);
                start = pos;
                gotDot = false;
            }
            else if (ch == 's')
            {
                if (gotS) return false;
                if (pos - 1 == start) return false;
                gotS = true;
                strS = str.substr(start, pos - 1 - start);
                start = pos;
                gotDot = false;
            }
        }
        if (pos != start) return false;
        if (!gotH && !gotM && !gotS) return false;

        uint64_t h = 0;
        if (gotH)
        {
            if (!parseU64(strH, &h)) return false;
        }

        uint64_t m = 0;
        if (gotM)
        {
            if (!parseU64(strM, &m)) return false;
            if (gotH && m >= 60) return false;
        }

        uint64_t s = 0;
        uint64_t ms = 0;
        if (gotS)
        {
            auto dotPos = strS.find('.');
            if (dotPos == string::npos)
            {
                if (!parseU64(strS, &s)) return false;
            }
            else
            {
                string strBefore = strS.substr(0, dotPos);
                string strAfter = strS.substr(dotPos + 1);
                if (!strBefore.empty())
                {
                    if (!parseU64(strBefore, &s)) return false;
                }
                if (strAfter.empty()) return false;
                if (strAfter.size() > 3) return false;
                if (strAfter.size() > 0) ms += 100 * (strAfter[0] - '0');
                if (strAfter.size() > 1) ms += 10 * (strAfter[1] - '0');
                if (strAfter.size() > 2) ms += 1 * (strAfter[2] - '0');
            }
            if (gotM && s >= 60) return false;
            if (gotH && s >= 3600) return false;
        }

        uint64_t limit = std::numeric_limits<uint64_t>::max();
        if (wantMS) limit /= 1000;

        uint64_t duration = 0;

        if (h > (limit - duration) / (60 * 60)) return false;
        duration += h * 60 * 60;

        if (m > (limit - duration) / 60) return false;
        duration += m * 60;

        if (s > (limit - duration)) return false;
        duration += s;

        if (wantMS)
        {
            duration *= 1000;
            limit = std::numeric_limits<uint64_t>::max();

            if (ms > (limit - duration)) return false;
            duration += ms;
        }

        if (pDuration) *pDuration = duration;
        return true;
    }
}

bool parseDurationS(const string& str, uint64_t* pDurationS)
{
    return parseDurationImpl(str, false, pDurationS);
}

bool parseDurationMS(const string& str, uint64_t* pDurationMS)
{
    return parseDurationImpl(str, true, pDurationMS);
}
