#include "Protocol.hpp"
#include "Http.hpp"
#include "FileUtils.hpp"
#include "StringUtils.hpp"
#include "Json.hpp"
#include "JsonUtils.hpp"
#include "Problem.hpp"
#include "Leaderboard.hpp"
#include "Solution.hpp"

using std::string;
using std::vector;
using std::pair;

namespace J = rapidjson;

namespace Protocol
{
    string apiUrl = "https://31pwr5t6ij.execute-api.eu-west-2.amazonaws.com/";

    void init()
    {
        Http::init();
    }

    void cleanup()
    {
        Http::cleanup();
    }

    bool getAPIKey(string* pKey,
                   string* pMsg = nullptr)
    {
        string tmpKey;
        if (!readFile("apikey.txt", &tmpKey) &&
            !readFile("../apikey.txt", &tmpKey) &&
            !readFile("../../apikey.txt", &tmpKey) &&
            !readFile("../../../apikey.txt", &tmpKey))
        {
            if (pMsg) *pMsg = "Error reading API key";
            return false;
        }

        string key;
        for (char ch : tmpKey)
        {
            //if (ch >= 33 && ch <= 126)
            if (ch >= 32 && ch <= 126)
            {
                key.push_back(ch);
            }
        }
        if (key.empty())
        {
            if (pMsg) *pMsg = "API key empty";
            return false;
        }

        if (pKey) *pKey = std::move(key);
        return true;
    }

    void getUserAgentHeader(vector<string>* pHeaders)
    {
        pHeaders->push_back("User-Agent: libcurl (Frictionless Bananas ICFPC 2025 tools)");
    }

    bool getProblems(vector<Problem>* pProblems,
                     string* pMsg)
    {
        vector<string> headers;
        getUserAgentHeader(&headers);

        string url = apiUrl + "select";

        string response;
        if (!Http::get(url, &headers, &response, pMsg))
        {
            return false;
        }

        vector<Problem> problems;

        try
        {
            J::Document jProblems;
            jProblems.Parse(response);
            JCheckParse(jProblems);
            JForEach(auto& jProblem, jProblems,
                     auto& problem, problems)
            {
                JCheckObject(jProblem);
                problem.name = JObjectGetString(jProblem, "problem");
                problem.size = JObjectGetUint64(jProblem, "size");
            }
        }
        catch (const JException& e)
        {
            if (pMsg) *pMsg = e.m_msg;
            return false;
        }

        if (pProblems) *pProblems = std::move(problems);
        return true;
    }

    bool getLeaderboard(const string& problemName,
                        vector<LeaderboardEntry>* pLeaderboard,
                        string* pMsg)
    {
        vector<string> headers;
        getUserAgentHeader(&headers);

        string url = apiUrl + "leaderboard/" + problemName;

        string response;
        if (!Http::get(url, &headers, &response, pMsg))
        {
            return false;
        }

        vector<LeaderboardEntry> leaderboard;

        try
        {
            J::Document jLeaderboard;
            jLeaderboard.Parse(response);
            JCheckParse(jLeaderboard);
            JForEach(auto& jEntry, jLeaderboard,
                     auto& entry, leaderboard)
            {
                JCheckObject(jEntry);
                entry.name = JObjectGetString(jEntry, "teamName");
                entry.pl = JObjectGetString(jEntry, "teamPl");
                auto& jScore = JObjectGet(jEntry, "score");
                if (jScore.IsNull())
                {
                    entry.score = 0;
                    entry.valid = false;
                }
                else if (jScore.IsUint64())
                {
                    entry.score = JGetUint64(jScore);
                    entry.valid = true;
                }
                else
                {
                    JError("jScore: not null or uint64");
                }
            }
        }
        catch (const JException& e)
        {
            if (pMsg) *pMsg = e.m_msg;
            return false;
        }

        if (pLeaderboard) *pLeaderboard = std::move(leaderboard);
        return true;
    }

    bool registerTeam(const string& name,
                      const string& pl,
                      const string& email,
                      string* pApiKey,
                      string* pMsg)
    {
        vector<string> headers;
        getUserAgentHeader(&headers);

        string url = apiUrl + "register";

        string request;

        {
            J::StringBuffer buffer;
            J::Writer<J::StringBuffer> w(buffer);

            w.StartObject();

            w.Key("name");
            w.String(name);

            w.Key("pl");
            w.String(pl);

            w.Key("email");
            w.String(email);

            w.EndObject();

            request = buffer.GetString();
        }

        string response;
        if (!Http::post(url, request, "application/json", &headers, &response, pMsg))
        {
            return false;
        }

        string apiKey;

        try
        {
            J::Document jResponse;
            jResponse.Parse(response);
            JCheckParse(jResponse);
            JCheckObject(jResponse);
            apiKey = JObjectGetString(jResponse, "id");
        }
        catch (const JException& e)
        {
            if (pMsg) *pMsg = e.m_msg;
            return false;
        }

        if (pApiKey) *pApiKey = std::move(apiKey);
        return true;
    }

    bool select(const string& problemName,
                string* pMsg)
    {
        string apiKey;
        if (!getAPIKey(&apiKey, pMsg)) return false;

        vector<string> headers;
        getUserAgentHeader(&headers);

        string url = apiUrl + "select";

        string request;

        {
            J::StringBuffer buffer;
            J::Writer<J::StringBuffer> w(buffer);

            w.StartObject();

            w.Key("id");
            w.String(apiKey);

            w.Key("problemName");
            w.String(problemName);

            w.EndObject();

            request = buffer.GetString();
        }

        string response;
        if (!Http::post(url, request, "application/json", &headers, &response, pMsg))
        {
            return false;
        }

        string responseProblemName;

        try
        {
            J::Document jResponse;
            jResponse.Parse(response);
            JCheckParse(jResponse);
            JCheckObject(jResponse);
            responseProblemName = JObjectGetString(jResponse, "problemName");
        }
        catch (const JException& e)
        {
            if (pMsg) *pMsg = e.m_msg;
            return false;
        }

        if (responseProblemName != problemName)
        {
            if (pMsg) *pMsg = "Response problems name does not match";
            return false;
        }

        return true;
    }

    bool explore(const vector<string>& plans,
                 vector<vector<uint8_t>>* pResults,
                 uint64_t* pQueryCount,
                 string* pMsg)
    {
        string apiKey;
        if (!getAPIKey(&apiKey, pMsg)) return false;

        vector<string> headers;
        getUserAgentHeader(&headers);

        string url = apiUrl + "explore";

        string request;

        {
            J::StringBuffer buffer;
            J::Writer<J::StringBuffer> w(buffer);

            w.StartObject();

            w.Key("id");
            w.String(apiKey);

            w.Key("plans");
            w.StartArray();
            for (auto& plan : plans)
            {
                w.String(plan);
            }
            w.EndArray();

            w.EndObject();

            request = buffer.GetString();
        }

        string response;
        if (!Http::post(url, request, "application/json", &headers, &response, pMsg))
        {
            return false;
        }

        vector<vector<uint8_t>> results;
        uint64_t queryCount = 0;

        try
        {
            J::Document jResponse;
            jResponse.Parse(response);
            JCheckParse(jResponse);
            JCheckObject(jResponse);

            auto& jResults = JObjectGetArray(jResponse, "results");
            JForEach(auto& jResult, jResults,
                     auto& result, results)
            {
                JCheckArray(jResult);
                JForEach(auto& jValue, jResult,
                         auto& value, result)
                {
                    uint64_t value64 = JGetUint64(jValue);
                    if (value64 > 3)
                    {
                        JError("jValue: value out of range");
                    }
                    value = (uint8_t)value64;
                }
            }

            queryCount = JObjectGetUint64(jResponse, "queryCount");
        }
        catch (const JException& e)
        {
            if (pMsg) *pMsg = e.m_msg;
            return false;
        }

        if (pResults) *pResults = std::move(results);
        if (pQueryCount) *pQueryCount = std::move(queryCount);
        return true;
    }

    bool guess(const Solution& solution,
               bool* pCorrect,
               string* pMsg)
    {
        string apiKey;
        if (!getAPIKey(&apiKey, pMsg)) return false;

        vector<string> headers;
        getUserAgentHeader(&headers);

        string url = apiUrl + "guess";

        string request;

        {
            J::StringBuffer buffer;
            J::Writer<J::StringBuffer> w(buffer);

            w.StartObject();

            w.Key("id");
            w.String(apiKey);

            w.Key("map");
            w.StartObject();
            {
                w.Key("rooms");
                w.StartArray();
                for (uint8_t value : solution.rooms)
                {
                    w.Uint64((uint64_t)value);
                }
                w.EndArray();

                w.Key("startingRoom");
                w.Uint64(solution.startingRoom);

                w.Key("connections");
                w.StartArray();
                for (auto& connection : solution.connections)
                {
                    w.StartObject();

                    w.Key("from");
                    w.StartObject();
                    {
                        w.Key("room");
                        w.Uint64(connection.from.room);

                        w.Key("door");
                        w.Uint64(connection.from.door);
                    }
                    w.EndObject();

                    w.Key("to");
                    w.StartObject();
                    {
                        w.Key("room");
                        w.Uint64(connection.to.room);

                        w.Key("door");
                        w.Uint64(connection.to.door);
                    }
                    w.EndObject();

                    w.EndObject();
                }
                w.EndArray();
            }
            w.EndObject();

            w.EndObject();

            request = buffer.GetString();
        }

        string response;
        if (!Http::post(url, request, "application/json", &headers, &response, pMsg))
        {
            return false;
        }

        bool correct = false;

        try
        {
            J::Document jResponse;
            jResponse.Parse(response);
            JCheckParse(jResponse);
            JCheckObject(jResponse);
            correct = JObjectGetBool(jResponse, "correct");
        }
        catch (const JException& e)
        {
            if (pMsg) *pMsg = e.m_msg;
            return false;
        }

        if (pCorrect) *pCorrect = std::move(correct);
        return true;
    }

#if 0
    bool getProblemCount(string* pProblemCountData,
                         string* pMsg)
    {
        vector<string> headers;
        getUserAgentHeader(&headers);

        string url = "https://api.icfpcontest.com/problems";

        string response;
        if (!Http::get(url, &headers, &response, pMsg))
        {
            return false;
        }

        if (pProblemCountData) *pProblemCountData = std::move(response);
        return true;
    }

    bool getProblem(uint64_t id,
                    string* pProblemData,
                    string* pMsg)
    {
        vector<string> headers;
        getUserAgentHeader(&headers);

        string url = strprintf("https://cdn.icfpcontest.com/problems/%" PRIu64 ".json", id);

        string response;
        if (!Http::get(url, &headers, &response, pMsg))
        {
            return false;
        }

        if (pProblemData) *pProblemData = std::move(response);
        return true;
    }

    bool getScoreboard(string* pScoreboardData,
                       string* pMsg)
    {
        vector<string> headers;
        getUserAgentHeader(&headers);

        string url = "https://api.icfpcontest.com/scoreboard";

        string response;
        if (!Http::get(url, &headers, &response, pMsg))
        {
            return false;
        }

        if (pScoreboardData) *pScoreboardData = std::move(response);
        return true;
    }

    bool getUserboard(string* pUserboardData,
                      string* pMsg)
    {
        vector<string> headers;
        if (!getAPIKeyHeader(&headers, pMsg)) return false;
        getUserAgentHeader(&headers);

        string url = "https://api.icfpcontest.com/userboard";

        string response;
        if (!Http::get(url, &headers, &response, pMsg))
        {
            return false;
        }

        if (pUserboardData) *pUserboardData = std::move(response);
        return true;
    }

    bool getSubmissions(uint64_t offset,
                        uint64_t limit,
                        string* pSubmissionsData,
                        string* pMsg)
    {
        return getSubmissions(offset, limit, (uint64_t)-1, pSubmissionsData, pMsg);
    }

    bool getSubmissions(uint64_t offset,
                        uint64_t limit,
                        uint64_t id,
                        string* pSubmissionsData,
                        string* pMsg)
    {
        vector<string> headers;
        if (!getAPIKeyHeader(&headers, pMsg)) return false;
        getUserAgentHeader(&headers);

        string url = "https://api.icfpcontest.com/submissions";

        vector<pair<string, string>> query;
        query.emplace_back("offset", strprintf("%" PRIu64 "", offset));
        query.emplace_back("limit", strprintf("%" PRIu64 "", limit));
        if (id != (uint64_t)-1)
        {
            query.emplace_back("problem_id", strprintf("%" PRIu64 "", id));
        }

        string response;
        if (!Http::getQuery(url, query, &headers, &response, pMsg))
        {
            return false;
        }

        if (pSubmissionsData) *pSubmissionsData = std::move(response);
        return true;
    }

    bool getSubmission(const string& submissionId,
                       string* pSubmissionData,
                       string* pMsg)
    {
        vector<string> headers;
        if (!getAPIKeyHeader(&headers, pMsg)) return false;
        getUserAgentHeader(&headers);

        string url = "https://api.icfpcontest.com/submission";

        vector<pair<string, string>> query;
        query.emplace_back("submission_id", submissionId);

        string response;
        if (!Http::getQuery(url, query, &headers, &response, pMsg))
        {
            return false;
        }

        if (pSubmissionData) *pSubmissionData = std::move(response);
        return true;
    }

    bool submit(uint64_t id,
                const string& solutionData,
                string* pSubmissionId,
                string* pMsg)
    {
        vector<string> headers;
        if (!getAPIKeyHeader(&headers, pMsg)) return false;
        getUserAgentHeader(&headers);

        string url = "https://api.icfpcontest.com/submission";

        string request;

        {
            J::StringBuffer buffer;
            J::Writer<J::StringBuffer> w(buffer);

            w.StartObject();

            w.Key("problem_id");
            w.Uint64(id);

            w.Key("contents");
            w.String(solutionData);

            w.EndObject();

            request = buffer.GetString();
        }

        string response;
        if (!Http::post(url, request, "application/json", &headers, &response, pMsg))
        {
            return false;
        }

        string submissionId;
        try
        {
            J::Document jSubmissionId;
            jSubmissionId.Parse(response);
            JCheckParse(jSubmissionId);
            submissionId = JGetString(jSubmissionId);
        }
        catch (const JException& e)
        {
            if (pMsg) *pMsg = e.m_msg;
            return false;
        }

        if (pSubmissionId) *pSubmissionId = std::move(submissionId);
        return true;
    }

    bool updateUsername(const string& username,
                        string* pMsg)
    {
        vector<string> headers;
        if (!getAPIKeyHeader(&headers, pMsg)) return false;
        getUserAgentHeader(&headers);

        string url = "https://api.icfpcontest.com/user/update_username";

        string request;

        {
            J::StringBuffer buffer;
            J::Writer<J::StringBuffer> w(buffer);

            w.StartObject();

            w.Key("username");
            w.String(username);

            w.EndObject();

            request = buffer.GetString();
        }

        string response;
        if (!Http::post(url, request, "application/json", &headers, &response, pMsg))
        {
            return false;
        }

        return true;
    }
#endif
}
