#include "Common.hpp"
#include "ParseUtils.hpp"
#include "Cleanup.hpp"
#include "Protocol.hpp"
#include "Solution.hpp"
#include "Xoshiro.hpp"
#include "FastRandom.hpp"
#include "TimeUtils.hpp"
#include "ProcessId.hpp"
#include <array>

using std::string;
using std::vector;
using std::array;

typedef Xoshiro256StarStar Gen;

class Target
{
public:
    uint32_t uniqueIdx = 0;
    uint8_t doorIdx = 0;
    uint8_t dupIdxs[3] = {};
};

class Room
{
public:
    uint8_t uniqueIdx = 0;
    uint8_t dupIdx = 0;
};

class UndoRec
{
public:
    Target from;
    Target to;
};

class TestMove
{
public:
    TestMove() : testIdx(0), moveIdx(0) {}
    TestMove(uint64_t testIdx_, uint64_t moveIdx_) : testIdx(testIdx_), moveIdx(moveIdx_) {}

    bool operator==(const TestMove& other) const
    {
        return testIdx == other.testIdx && moveIdx == other.moveIdx;
    }

    bool operator!=(const TestMove& other) const
    {
        return testIdx != other.testIdx || moveIdx != other.moveIdx;
    }

    uint64_t testIdx = 0;
    uint64_t moveIdx = 0;
};

void usage()
{
    printf("Usage: anneal <dup count> <unique count>\n");
    printf("  <dup count>\n");
    printf("        Number of duplicate copy of each distinguishable room type\n");
    printf("  <unique count>\n");
    printf("        Number of distinguishable room types\n");
    printf("Options:\n");
    printf("  -h    Print usage information and exit\n");
    printf("  -d    Dry run: don't actually submit\n");
}

class Solver
{
public:
    Solver(bool dryRun_, uint64_t dupCount_, uint64_t uniqueCount_) :
        dryRun(dryRun_), dupCount(dupCount_), uniqueCount(uniqueCount_)
    {
    }

    void setDups(uint64_t uniqueIdxI, uint64_t doorIdxI)
    {
        auto& targetI = connections[uniqueIdxI][doorIdxI];
        uint64_t uniqueIdxJ = targetI.uniqueIdx;
        uint64_t doorIdxJ = targetI.doorIdx;
        if (uniqueIdxJ == uniqueIdxI && doorIdxJ == doorIdxI)
        {
            switch (dupCount)
            {
            case 1:
                targetI.dupIdxs[0] = 0;
                break;
            case 2:
                switch (fastRandom(2, gen))
                {
                case 0:
                    targetI.dupIdxs[0] = 0;
                    targetI.dupIdxs[1] = 1;
                    break;
                case 1:
                    targetI.dupIdxs[0] = 1;
                    targetI.dupIdxs[1] = 0;
                    break;
                default: ;
                }
                break;
            case 3:
                switch (fastRandom(4, gen))
                {
                case 0:
                    targetI.dupIdxs[0] = 0;
                    targetI.dupIdxs[1] = 1;
                    targetI.dupIdxs[2] = 2;
                    break;
                case 1:
                    targetI.dupIdxs[0] = 1;
                    targetI.dupIdxs[1] = 0;
                    targetI.dupIdxs[2] = 2;
                    break;
                case 2:
                    targetI.dupIdxs[0] = 0;
                    targetI.dupIdxs[1] = 2;
                    targetI.dupIdxs[2] = 1;
                    break;
                case 3:
                    targetI.dupIdxs[0] = 2;
                    targetI.dupIdxs[1] = 1;
                    targetI.dupIdxs[2] = 0;
                    break;
                default: ;
                }
                break;
            default: ;
            }
        }
        else if (uniqueIdxJ > uniqueIdxI ||
                 (uniqueIdxJ == uniqueIdxI && doorIdxJ > doorIdxI))
        {
            auto& targetJ = connections[uniqueIdxJ][doorIdxJ];
            switch (dupCount)
            {
            case 1:
                targetI.dupIdxs[0] = 0; targetJ.dupIdxs[0] = 0;
                break;
            case 2:
                switch (fastRandom(2, gen))
                {
                case 0:
                    targetI.dupIdxs[0] = 0; targetJ.dupIdxs[0] = 0;
                    targetI.dupIdxs[1] = 1; targetJ.dupIdxs[1] = 1;
                    break;
                case 1:
                    targetI.dupIdxs[0] = 1; targetJ.dupIdxs[1] = 0;
                    targetI.dupIdxs[1] = 0; targetJ.dupIdxs[0] = 1;
                    break;
                default: ;
                }
                break;
            case 3:
                switch (fastRandom(6, gen))
                {
                case 0:
                    targetI.dupIdxs[0] = 0; targetJ.dupIdxs[0] = 0;
                    targetI.dupIdxs[1] = 1; targetJ.dupIdxs[1] = 1;
                    targetI.dupIdxs[2] = 2; targetJ.dupIdxs[2] = 2;
                    break;
                case 1:
                    targetI.dupIdxs[0] = 1; targetJ.dupIdxs[1] = 0;
                    targetI.dupIdxs[1] = 0; targetJ.dupIdxs[0] = 1;
                    targetI.dupIdxs[2] = 2; targetJ.dupIdxs[2] = 2;
                    break;
                case 2:
                    targetI.dupIdxs[0] = 0; targetJ.dupIdxs[0] = 0;
                    targetI.dupIdxs[1] = 2; targetJ.dupIdxs[2] = 1;
                    targetI.dupIdxs[2] = 1; targetJ.dupIdxs[1] = 2;
                    break;
                case 3:
                    targetI.dupIdxs[0] = 2; targetJ.dupIdxs[2] = 0;
                    targetI.dupIdxs[1] = 1; targetJ.dupIdxs[1] = 1;
                    targetI.dupIdxs[2] = 0; targetJ.dupIdxs[0] = 2;
                    break;
                case 4:
                    targetI.dupIdxs[0] = 1; targetJ.dupIdxs[1] = 0;
                    targetI.dupIdxs[1] = 2; targetJ.dupIdxs[2] = 1;
                    targetI.dupIdxs[2] = 0; targetJ.dupIdxs[0] = 2;
                    break;
                case 5:
                    targetI.dupIdxs[0] = 2; targetJ.dupIdxs[2] = 0;
                    targetI.dupIdxs[1] = 0; targetJ.dupIdxs[0] = 1;
                    targetI.dupIdxs[2] = 1; targetJ.dupIdxs[1] = 2;
                    break;
                default: ;
                }
                break;
            default: ;
            }
        }
    }

    int solve()
    {
        switch (dupCount)
        {
        case 1:
            switch (uniqueCount)
            {
            case 3: problemName = "probatio"; break;
            case 6: problemName = "primus"; break;
            case 12: problemName = "secundus"; break;
            case 18: problemName = "tertius"; break;
            case 24: problemName = "quartus"; break;
            case 30: problemName = "quintus"; break;
            default:
                printf("Unknown unique count\n");
                return 1;
            }
            break;
        case 2:
            switch (uniqueCount)
            {
            case 6: problemName = "aleph"; break;
            case 12: problemName = "beth"; break;
            case 18: problemName = "gimel"; break;
            case 24: problemName = "daleth"; break;
            case 30: problemName = "he"; break;
            default:
                printf("Unknown unique count\n");
                return 1;
            }
            break;
        case 3:
            switch (uniqueCount)
            {
            case 6: problemName = "vau"; break;
            case 12: problemName = "zain"; break;
            case 18: problemName = "hhet"; break;
            case 24: problemName = "teth"; break;
            case 30: problemName = "iod"; break;
            default:
                printf("Unknown unique count\n");
                return 1;
            }
            break;
        default:
            printf("Unknown dup count\n");
            return 1;
        }

        roomCount = dupCount * uniqueCount;

        printf("dup count: %" PRIu64 "\n", dupCount);
        printf("unique count: %" PRIu64 "\n", uniqueCount);
        printf("room count: %" PRIu64 "\n", roomCount);
        printf("problem: %s\n", problemName.c_str());

        Protocol::init();
        Cleanup cleanupProtocol([](){ Protocol::cleanup(); });

        string msg;

        if (!Protocol::select(problemName, &msg))
        {
            printf("%s\n", msg.c_str());
            return 1;
        }

#if 1
        {
            uint64_t timeMS = getTimeMS();
            uint32_t pid = getProcessId();
            uint32_t seed0 = 0xddf294ea;
            uint32_t seed1 = (uint32_t)timeMS;
            uint32_t seed2 = (uint32_t)pid ^ (uint32_t)(timeMS >> 32);
            std::seed_seq seq{seed0, seed1, seed2};
            gen.seed(seq);
        }
#endif
#if 0
        {
            uint32_t seed0 = 0xddf294ea;
            std::seed_seq seq{seed0};
            gen.seed(seq);
        }
#endif

        if (dupCount == 1)
        {
            testCount = 1;
            moveCount = roomCount * 18;
            moveSplit = roomCount * 6;
        }
        else
        {
            testCount = 1;
            moveCount = roomCount * 6;
            moveSplit = roomCount * 2;
        }

        planCount = testCount * 2;

        moves.resize(planCount);
        marks.resize(planCount);
        for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
        {
            moves[planIdx].resize(moveCount);
            marks[planIdx].resize(moveCount);
            for (uint64_t moveIdx = 0; moveIdx < moveCount; moveIdx++)
            {
                if ((planIdx % 2) == 0)
                {
                    moves[planIdx][moveIdx] = fastRandom((uint32_t)6, gen);
                    marks[planIdx][moveIdx] = 4;
                }
                else
                {
                    moves[planIdx][moveIdx] = moves[planIdx - 1][moveIdx];
                    if (moveIdx < moveSplit)
                    {
                        marks[planIdx][moveIdx] = fastRandom((uint32_t)4, gen);
                    }
                    else
                    {
                        marks[planIdx][moveIdx] = 4;
                    }
                }
            }
        }

        vector<string> plans(planCount);
        for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
        {
            auto& plan = plans[planIdx];
            for (uint64_t moveIdx = 0; moveIdx < moveCount; moveIdx++)
            {
                if (marks[planIdx][moveIdx] != 4)
                {
                    plan += '[';
                    plan += (char)('0' + marks[planIdx][moveIdx]);
                    plan += ']';
                }
                plan += (char)('0' + moves[planIdx][moveIdx]);
            }
        }

        for (auto& plan : plans)
        {
            printf("%s\n", plan.c_str());
        }

        vector<vector<uint8_t>> results;
        uint64_t queryCount;
        if (!Protocol::explore(plans, &results, &queryCount, &msg))
        {
            printf("%s\n", msg.c_str());
            return 1;
        }

        for (auto& result : results)
        {
            for (uint8_t value : result)
            {
                printf("%" PRIu8 "", value);
            }
            printf("\n");
        }
        printf("queryCount = %" PRIu64 "\n", queryCount);

        values.resize(planCount);
        for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
        {
            values[planIdx].resize(moveCount + 1);
            uint64_t resultIdx = 0;
            for (uint64_t moveIdx = 0; moveIdx <= moveCount; moveIdx++)
            {
                values[planIdx][moveIdx] = results[planIdx][resultIdx++];
                if (moveIdx < moveCount && marks[planIdx][moveIdx] != 4)
                {
                    resultIdx++;
                }
            }
        }

        while (true)
        {
            roomMap.resize(testCount);
            for (uint64_t testIdx = 0; testIdx < testCount; testIdx++)
            {
                roomMap[testIdx].resize(moveCount + 1);
                for (uint64_t moveIdx = 0; moveIdx <= moveCount; moveIdx++)
                {
                    auto& room = roomMap[testIdx][moveIdx];
                    uint64_t value = values[testIdx * 2][moveIdx];
                    uint64_t valueCount = (uniqueCount + 3 - value) / 4;
                    room.uniqueIdx = fastRandom(valueCount, gen) * 4 + value;
                    room.dupIdx = fastRandom(dupCount, gen);
                }
            }

            markMap.resize(testCount);
            for (uint64_t testIdx = 0; testIdx < testCount; testIdx++)
            {
                markMap[testIdx].resize(uniqueCount);
                for (uint64_t uniqueIdx = 0; uniqueIdx < uniqueCount; uniqueIdx++)
                {
                    for (uint64_t dupIdx = 0; dupIdx < dupCount; dupIdx++)
                    {
                        markMap[testIdx][uniqueIdx][dupIdx] = fastRandom((uint32_t)4, gen);
                    }
                }
            }

            connections.resize(uniqueCount);
            for (uint64_t uniqueIdx = 0; uniqueIdx < uniqueCount; uniqueIdx++)
            {
                for (uint64_t doorIdx = 0; doorIdx < 6; doorIdx++)
                {
                    auto& target = connections[uniqueIdx][doorIdx];
                    target.uniqueIdx = uniqueIdx;
                    target.doorIdx = doorIdx;
                    target.dupIdxs[0] = 0;
                    target.dupIdxs[1] = 1;
                    target.dupIdxs[2] = 2;
                }
            }

            for (uint64_t i = 0; i < uniqueCount * 6; i++)
            {
                uint64_t fromUniqueIdx = fastRandom(uniqueCount, gen);
                uint64_t fromDoorIdx = fastRandom(6, gen);
                auto& fromTarget = connections[fromUniqueIdx][fromDoorIdx];
                if (fromTarget.uniqueIdx != fromUniqueIdx ||
                    fromTarget.doorIdx != fromDoorIdx)
                {
                    continue;
                }

                uint64_t toUniqueIdx = fastRandom(uniqueCount, gen);
                uint64_t toDoorIdx = fastRandom(6, gen);
                if (toUniqueIdx == fromUniqueIdx && toDoorIdx == fromDoorIdx)
                {
                    continue;
                }
                auto& toTarget = connections[toUniqueIdx][toDoorIdx];
                if (toTarget.uniqueIdx != toUniqueIdx ||
                    toTarget.doorIdx != toDoorIdx)
                {
                    continue;
                }

                fromTarget.uniqueIdx = toUniqueIdx;
                fromTarget.doorIdx = toDoorIdx;
                toTarget.uniqueIdx = fromUniqueIdx;
                toTarget.doorIdx = fromDoorIdx;
            }

            for (uint64_t uniqueIdxI = 0; uniqueIdxI < uniqueCount; uniqueIdxI++)
            {
                for (uint64_t doorIdxI = 0; doorIdxI < 6; doorIdxI++)
                {
                    setDups(uniqueIdxI, doorIdxI);
                }
            }

#if 0
            vector<array<vector<TestMove>, 6>> doorMap(uniqueCount);
            for (uint64_t testIdx = 0; testIdx < testCount; testIdx++)
            {
                for (uint64_t moveIdx = 0; moveIdx < moveCount; moveIdx++)
                {
                    uint64_t uniqueIdx = roomMap[testIdx][moveIdx].uniqueIdx;
                    uint64_t doorIdx = moves[testIdx * 2][moveIdx];
                    doorMap[uniqueIdx][doorIdx].emplace_back(testIdx, moveIdx);
                }
            }
#endif

            auto calcCost = [&]() -> uint64_t
            {
                uint64_t cost = 0;
                for (uint64_t testIdx = 0; testIdx < testCount; testIdx++)
                {
                    for (uint64_t moveIdx = 0; moveIdx < moveCount; moveIdx++)
                    {
                        uint64_t fromUniqueIdx = roomMap[testIdx][moveIdx].uniqueIdx;
                        uint64_t fromDupIdx = roomMap[testIdx][moveIdx].dupIdx;
                        uint64_t toUniqueIdx = roomMap[testIdx][moveIdx + 1].uniqueIdx;
                        uint64_t toDupIdx = roomMap[testIdx][moveIdx + 1].dupIdx;
                        uint64_t doorIdx = moves[testIdx * 2][moveIdx];
                        if (connections[fromUniqueIdx][doorIdx].uniqueIdx != toUniqueIdx ||
                            (moveIdx >= moveSplit && connections[fromUniqueIdx][doorIdx].dupIdxs[fromDupIdx] != toDupIdx))
                        {
                            cost++;
                        }
                    }

                    for (uint64_t moveIdx = moveSplit; moveIdx <= moveCount; moveIdx++)
                    {
                        uint64_t uniqueIdx = roomMap[testIdx][moveIdx].uniqueIdx;
                        uint64_t dupIdx = roomMap[testIdx][moveIdx].dupIdx;
                        uint64_t mark = markMap[testIdx][uniqueIdx][dupIdx];
                        if (values[testIdx * 2 + 1][moveIdx] != mark)
                        {
                            cost++;
                        }
                    }
                }
                return cost;
            };

            uint64_t cost = calcCost();
            uint64_t bestCost = cost;
            printf("%9" PRIu64 "  %4" PRIu64 "\n", (uint64_t)0, cost);

            double allowedLossInitial = 0.8;
            double allowedLossFinal = 0.2;
            double lnRatio = std::log(allowedLossFinal / allowedLossInitial);

            double allowedLoss = allowedLossInitial;
            double recipAllowedLoss = 1.0 / allowedLoss;

            vector<UndoRec> undoLog;

            uint64_t iterLimit = uniqueCount * uniqueCount * 1000000;

            uint64_t iter = 0;
            while (iter < iterLimit)
            {
                iter++;

                if ((iter & 0xf) == 0)
                {
                    double param = (double)iter / (double)iterLimit;
                    allowedLoss = allowedLossInitial * std::exp(lnRatio * param);
                    recipAllowedLoss = 1.0 / allowedLoss;
                }

                uint64_t changeType = fastRandom((uint64_t)3, gen);
                uint64_t testIdx = (uint64_t)-1;
                uint64_t moveIdx = (uint64_t)-1;
                uint64_t oldUniqueIdx = (uint64_t)-1;
                uint64_t oldDupIdx = (uint64_t)-1;
                uint64_t newUniqueIdx = (uint64_t)-1;
                uint64_t newDupIdx = (uint64_t)-1;
                uint64_t markUniqueIdx = (uint64_t)-1;
                uint64_t markDupIdx = (uint64_t)-1;
                uint64_t oldMark = (uint64_t)1;

                uint64_t curCost = cost;

                if (changeType == 0)
                {
                    testIdx = fastRandom(testCount, gen);
                    moveIdx = fastRandom(moveCount + 1, gen);
                    oldUniqueIdx = roomMap[testIdx][moveIdx].uniqueIdx;
                    oldDupIdx = roomMap[testIdx][moveIdx].dupIdx;
                    uint64_t value = values[testIdx * 2][moveIdx];
                    uint64_t valueCount = (uniqueCount + 3 - value) / 4;
                    newUniqueIdx = fastRandom(valueCount, gen) * 4 + value;
                    if (moveIdx < moveSplit)
                    {
                        newDupIdx = oldDupIdx;
                    }
                    else
                    {
                        newDupIdx = fastRandom(dupCount, gen);
                    }
                    if (newUniqueIdx == oldUniqueIdx &&
                        newDupIdx == oldDupIdx)
                    {
                        continue;
                    }
                    roomMap[testIdx][moveIdx].uniqueIdx = newUniqueIdx;
                    roomMap[testIdx][moveIdx].dupIdx = newDupIdx;

#if 0
                    if (moveIdx > 0)
                    {
                        uint64_t prevUniqueIdx = roomMap[planIdx][moveIdx - 1].uniqueIdx;
                        uint64_t doorIdx = moves[planIdx][moveIdx - 1];
                        if (connections[prevUniqueIdx][doorIdx].uniqueIdx != oldUniqueIdx)
                        {
                            curCost--;
                        }
                        if (connections[prevUniqueIdx][doorIdx].uniqueIdx != newUniqueIdx)
                        {
                            curCost++;
                        }
                    }

                    if (moveIdx < moveSplit)
                    {
                        uint64_t nextUniqueIdx = roomMap[planIdx][moveIdx + 1].uniqueIdx;
                        uint64_t doorIdx = moves[planIdx][moveIdx];
                        if (connections[oldUniqueIdx][doorIdx].uniqueIdx != nextUniqueIdx)
                        {
                            curCost--;
                        }
                        if (connections[newUniqueIdx][doorIdx].uniqueIdx != nextUniqueIdx)
                        {
                            curCost++;
                        }
                    }
#endif

                    curCost = calcCost();
                }
                else if (changeType == 1)
                {
                    testIdx = fastRandom(testCount, gen);
                    markUniqueIdx = fastRandom(uniqueCount, gen);
                    markDupIdx = fastRandom(dupCount, gen);
                    oldMark = markMap[testIdx][markUniqueIdx][markDupIdx];
                    markMap[testIdx][markUniqueIdx][markDupIdx] = fastRandom((uint32_t)4, gen);

                    curCost = calcCost();
                }
                else if (changeType == 2)
                {
                    uint64_t fromUniqueIdx = fastRandom(uniqueCount, gen);
                    uint64_t fromDoorIdx = fastRandom(6, gen);
                    auto fromTarget = connections[fromUniqueIdx][fromDoorIdx];

                    uint64_t toUniqueIdx = fastRandom(uniqueCount, gen);
                    uint64_t toDoorIdx = fastRandom(6, gen);
                    auto toTarget = connections[toUniqueIdx][toDoorIdx];

                    undoLog.clear();

                    {
                        auto& rec = undoLog.emplace_back();
                        rec.from = connections[fromTarget.uniqueIdx][fromTarget.doorIdx];
                        rec.to = fromTarget;
                    }

                    {
                        auto& rec = undoLog.emplace_back();
                        rec.from = connections[toTarget.uniqueIdx][toTarget.doorIdx];
                        rec.to = toTarget;
                    }

#if 0
                    auto updateCost = [&](uint64_t fromUniqueIdx,
                                          uint64_t fromDoorIdx,
                                          uint64_t oldToUniqueIdx,
                                          uint64_t newToUniqueIdx)
                    {
                        for (auto& planMove : doorMap[fromUniqueIdx][fromDoorIdx])
                        {
                            if (oldToUniqueIdx != roomMap[planMove.planIdx][planMove.moveIdx + 1].uniqueIdx)
                            {
                                curCost--;
                            }
                            if (newToUniqueIdx != roomMap[planMove.planIdx][planMove.moveIdx + 1].uniqueIdx)
                            {
                                curCost++;
                            }
                        }
                    };
#endif

                    if (fromUniqueIdx != toUniqueIdx ||
                        fromDoorIdx != toDoorIdx)
                    {
                        if ((fromTarget.uniqueIdx != fromUniqueIdx ||
                             fromTarget.doorIdx != fromDoorIdx) &&
                            (toTarget.uniqueIdx != toUniqueIdx ||
                             toTarget.doorIdx != toDoorIdx))
                        {
                            connections[fromTarget.uniqueIdx][fromTarget.doorIdx] = toTarget;
                            connections[toTarget.uniqueIdx][toTarget.doorIdx] = fromTarget;

                            setDups(fromTarget.uniqueIdx, fromTarget.doorIdx);

                            //updateCost(fromTarget.uniqueIdx, fromTarget.doorIdx, fromUniqueIdx, toTarget.uniqueIdx);
                            //updateCost(toTarget.uniqueIdx, toTarget.doorIdx, toUniqueIdx, fromTarget.uniqueIdx);
                        }
                        else
                        {
                            if (fromTarget.uniqueIdx != fromUniqueIdx ||
                                fromTarget.doorIdx != fromDoorIdx)
                            {
                                connections[fromTarget.uniqueIdx][fromTarget.doorIdx] = fromTarget;
                                setDups(fromTarget.uniqueIdx, fromTarget.doorIdx);
                                //updateCost(fromTarget.uniqueIdx, fromTarget.doorIdx, fromUniqueIdx, fromTarget.uniqueIdx);
                            }

                            if (toTarget.uniqueIdx != toUniqueIdx ||
                                toTarget.doorIdx != toDoorIdx)
                            {
                                connections[toTarget.uniqueIdx][toTarget.doorIdx] = toTarget;
                                setDups(toTarget.uniqueIdx, toTarget.doorIdx);
                                //updateCost(toTarget.uniqueIdx, toTarget.doorIdx, toUniqueIdx, toTarget.uniqueIdx);
                            }
                        }
                    }

                    connections[fromUniqueIdx][fromDoorIdx].uniqueIdx = toUniqueIdx;
                    connections[fromUniqueIdx][fromDoorIdx].doorIdx = toDoorIdx;
                    connections[toUniqueIdx][toDoorIdx].uniqueIdx = fromUniqueIdx;
                    connections[toUniqueIdx][toDoorIdx].doorIdx = fromDoorIdx;

                    setDups(fromUniqueIdx, fromDoorIdx);

                    //updateCost(fromUniqueIdx, fromDoorIdx, fromTarget.uniqueIdx, toUniqueIdx);
                    //updateCost(toUniqueIdx, toDoorIdx, toTarget.uniqueIdx, fromUniqueIdx);

                    curCost = calcCost();
                }

#if 0
                uint64_t slowCost = calcCost();
                if (slowCost != curCost)
                {
                    printf("Incremental cost error!\n");
                    return 1;
                }
#endif

                bool keep;

                if (curCost <= cost)
                {
                    keep = true;
                }
                else
                {
                    double loss = (double)(curCost - cost);
#if 0
                    keep = false;
#endif
#if 1
                    keep = fastRandomDouble(gen) < std::exp(-loss * recipAllowedLoss);
#endif
#if 0
                    keep = fastRandomDouble(gen) >= loss * recipAllowedLoss;
#endif
                }

                if (keep)
                {
                    cost = curCost;
                    if (cost < bestCost)
                    {
                        printf("%9" PRIu64 "  %4" PRIu64 "\n", iter, cost);
                        bestCost = cost;
                    }

                    if (changeType == 0)
                    {
#if 0
                        if (moveIdx < moveSplit)
                        {
                            uint64_t doorIdx = moves[planIdx][moveIdx];
                            std::erase(doorMap[oldUniqueIdx][doorIdx], PlanMove(planIdx, moveIdx));
                            doorMap[newUniqueIdx][doorIdx].emplace_back(planIdx, moveIdx);
                        }
#endif

#if 0
                        for (uint64_t uniqueIdx = 0; uniqueIdx < uniqueCount; uniqueIdx++)
                        {
                            for (uint64_t doorIdx = 0; doorIdx < 6; doorIdx++)
                            {
                                for (auto& planMove : doorMap[uniqueIdx][doorIdx])
                                {
                                    if (roomMap[planMove.planIdx][planMove.moveIdx].uniqueIdx != uniqueIdx ||
                                        moves[planMove.planIdx][planMove.moveIdx] != doorIdx)
                                    {
                                        printf("doorMap error! (1)\n");
                                        return 1;
                                    }
                                }
                            }
                        }

                        for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
                        {
                            for (uint64_t moveIdx = 0; moveIdx < moveSplit; moveIdx++)
                            {
                                uint64_t uniqueIdx = roomMap[planIdx][moveIdx].uniqueIdx;
                                uint64_t doorIdx = moves[planIdx][moveIdx];
                                if (std::find(doorMap[uniqueIdx][doorIdx].begin(),
                                              doorMap[uniqueIdx][doorIdx].end(),
                                              PlanMove(planIdx, moveIdx)) ==
                                    doorMap[uniqueIdx][doorIdx].end())
                                {
                                    printf("doorMap error! (2)\n");
                                    return 1;
                                }
                            }
                        }
#endif
                    }
                }
                else
                {
                    if (changeType == 0)
                    {
                        roomMap[testIdx][moveIdx].uniqueIdx = oldUniqueIdx;
                        roomMap[testIdx][moveIdx].dupIdx = oldDupIdx;
                    }
                    else if (changeType == 1)
                    {
                        markMap[testIdx][markUniqueIdx][markDupIdx] = oldMark;
                    }
                    else if (changeType == 2)
                    {
                        for (auto& rec : undoLog)
                        {
                            connections[rec.from.uniqueIdx][rec.from.doorIdx] = rec.to;
                            connections[rec.to.uniqueIdx][rec.to.doorIdx] = rec.from;
                        }
                    }
                }

                if (bestCost == 0) break;
            }

            if (bestCost != 0) continue;

#if 0
            for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
            {
                for (uint64_t moveIdx = moveSplit; moveIdx < moveCount; moveIdx++)
                {
                    uint64_t fromUniqueIdx = roomMap[planIdx][moveIdx].uniqueIdx;
                    uint64_t doorIdx = moves[planIdx][moveIdx];
                    uint64_t toUniqueIdx = connections[fromUniqueIdx][doorIdx].uniqueIdx;
                    roomMap[planIdx][moveIdx + 1].uniqueIdx = toUniqueIdx;
                }
            }

            roomMarks.resize(planCount);
            for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
            {
                roomMarks[planIdx].resize(uniqueCount);
                for (uint64_t uniqueIdx = 0; uniqueIdx < uniqueCount; uniqueIdx++)
                {
                    for (uint64_t dupIdx = 0; dupIdx < 3; dupIdx++)
                    {
                        uint64_t value = uniqueIdx % 4;
                        roomMarks[planIdx][uniqueIdx][dupIdx] = value;
                    }
                }
            }

            bool dupsResult = solveDups((uint64_t)-1, moveCount);

            printf("dupsResult = %s\n", dupsResult ? "true" : "false");

            if (!dupsResult) continue;

            uint64_t unassignedHist[4] = {};

            for (uint64_t fromUniqueIdx = 0; fromUniqueIdx < uniqueCount; fromUniqueIdx++)
            {
                for (uint64_t fromDoorIdx = 0; fromDoorIdx < 6; fromDoorIdx++)
                {
                    auto& fromTarget = connections[fromUniqueIdx][fromDoorIdx];
                    uint64_t toUniqueIdx = fromTarget.uniqueIdx;
                    uint64_t toDoorIdx = fromTarget.doorIdx;
                    auto& toTarget = connections[toUniqueIdx][toDoorIdx];
                    uint64_t unassignedCount = 0;
                    for (uint64_t fromDupIdx = 0; fromDupIdx < dupCount; fromDupIdx++)
                    {
                        if (fromTarget.dupIdxs[fromDupIdx] != 4) continue;
                        unassignedCount++;
                        bool found = false;
                        for (uint64_t toDupIdx = 0; toDupIdx < dupCount; toDupIdx++)
                        {
                            if (toTarget.dupIdxs[toDupIdx] != 4) continue;
                            fromTarget.dupIdxs[fromDupIdx] = toDupIdx;
                            toTarget.dupIdxs[toDupIdx] = fromDupIdx;
                            found = true;
                            break;
                        }
                        if (!found)
                        {
                            printf("Unassigned dup error\n");
                            return 1;
                        }
                    }
                    unassignedHist[unassignedCount]++;
                }
            }

            printf("Unassigned:");
            for (uint64_t unassignedCount = 0; unassignedCount <= dupCount; unassignedCount++)
            {
                printf(" %" PRIu64 "", unassignedHist[unassignedCount]);
            }
            printf("\n");

            uint64_t startUniqueIdx = roomMap[0][0].uniqueIdx;
            uint64_t startDupIdx = (uint64_t)-1;

            for (uint64_t curStartDupIdx = 0; curStartDupIdx < dupCount; curStartDupIdx++)
            {
                bool ok = true;

                for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
                {
                    roomMap[planIdx][0].dupIdx = curStartDupIdx;
                    for (uint64_t moveIdx = 0; moveIdx < moveSplit; moveIdx++)
                    {
                        uint64_t fromUniqueIdx = roomMap[planIdx][moveIdx].uniqueIdx;
                        uint64_t fromDupIdx = roomMap[planIdx][moveIdx].dupIdx;
                        uint64_t fromDoorIdx = moves[planIdx][moveIdx];
                        auto& fromTarget = connections[fromUniqueIdx][fromDoorIdx];
                        uint64_t toDupIdx = fromTarget.dupIdxs[fromDupIdx];

                        if (moveIdx + 1 < moveSplit)
                        {
                            roomMap[planIdx][moveIdx + 1].dupIdx = toDupIdx;
                        }
                        else
                        {
                            printf("curStartDupIdx %" PRIu64 "  planIdx %" PRIu64 "  toDupIdx %" PRIu64 "  expected toDupIdx %" PRIu64 "\n", curStartDupIdx, planIdx, toDupIdx, (uint64_t)roomMap[planIdx][moveIdx + 1].dupIdx);
                            if (roomMap[planIdx][moveIdx + 1].dupIdx != toDupIdx)
                            {
                                ok = false;
                            }
                        }
                    }
                    if (!ok) break;
                }

                if (ok)
                {
                    startDupIdx = curStartDupIdx;
                    break;
                }
            }

            if (startDupIdx == (uint64_t)-1)
            {
                printf("Failed to find startDupIdx\n");
                continue;
            }

            Solution solution;
            for (uint64_t uniqueIdx = 0; uniqueIdx < uniqueCount; uniqueIdx++)
            {
                for (uint64_t dupIdx = 0; dupIdx < dupCount; dupIdx++)
                {
                    uint64_t value = uniqueIdx % 4;
                    printf("%" PRIu64 ": %" PRIu64 "\n", uniqueIdx * dupCount + dupIdx, value);
                    solution.rooms.push_back(value);
                }
            }

            uint64_t startRoomIdx = startUniqueIdx * dupCount + startDupIdx;
            printf("start: %" PRIu64 "\n", startRoomIdx);
            solution.startingRoom = startRoomIdx;

            for (uint64_t fromUniqueIdx = 0; fromUniqueIdx < uniqueCount; fromUniqueIdx++)
            {
                for (uint64_t fromDoorIdx = 0; fromDoorIdx < 6; fromDoorIdx++)
                {
                    auto& fromTarget = connections[fromUniqueIdx][fromDoorIdx];
                    uint64_t toUniqueIdx = fromTarget.uniqueIdx;
                    uint64_t toDoorIdx = fromTarget.doorIdx;
                    for (uint64_t fromDupIdx = 0; fromDupIdx < dupCount; fromDupIdx++)
                    {
                        uint64_t toDupIdx = fromTarget.dupIdxs[fromDupIdx];
                        if ((fromUniqueIdx < toUniqueIdx) ||
                            (fromUniqueIdx == toUniqueIdx && fromDoorIdx < toDoorIdx) ||
                            (fromUniqueIdx == toUniqueIdx && fromDoorIdx == toDoorIdx && fromDupIdx <= toDupIdx))
                        {
                            uint64_t fromRoomIdx = fromUniqueIdx * dupCount + fromDupIdx;
                            uint64_t toRoomIdx = toUniqueIdx * dupCount + toDupIdx;

                            printf("(%" PRIu64 ", %" PRIu64 ") - (%" PRIu64 ", %" PRIu64 ")\n", fromRoomIdx, fromDoorIdx, toRoomIdx, toDoorIdx);

                            auto& connection = solution.connections.emplace_back();
                            connection.from.room = fromRoomIdx;
                            connection.from.door = fromDoorIdx;
                            connection.to.room = toRoomIdx;
                            connection.to.door = toDoorIdx;
                        }
                    }
                }
            }

            if (dryRun)
            {
                printf("Dry run: not submitting\n");
            }
            else
            {
                bool correct;
                if (!Protocol::guess(solution, &correct, &msg))
                {
                    printf("%s\n", msg.c_str());
                    return 1;
                }

                printf("correct = %s\n", correct ? "true" : "false");
            }
#endif

            break;
        }

        return 0;
    }

public:
    Gen gen;
    bool dryRun;
    uint64_t dupCount;
    uint64_t uniqueCount;
    uint64_t roomCount;
    string problemName;
    uint64_t testCount;
    uint64_t planCount;
    uint64_t moveCount;
    uint64_t moveSplit;
    vector<vector<uint8_t>> moves;
    vector<vector<uint8_t>> marks;
    vector<vector<uint8_t>> values;
    vector<vector<Room>> roomMap;
    vector<vector<array<uint8_t, 3>>> markMap;
    vector<array<Target, 6>> connections;
    //vector<vector<array<uint8_t, 3>>> roomMarks;
};

int main(int argc, char *argv[])
{
    bool gotDupCount = false;
    bool gotUniqueCount = false;

    bool help = false;
    bool dryRun = false;
    uint64_t dupCount = 0;
    uint64_t uniqueCount = 0;

    int iArg = 1;
    while (iArg < argc)
    {
        string strArg = argv[iArg++];

        if (strArg == "-h" || strArg == "--help")
        {
            help = true;
        }
        else if (strArg == "-d")
        {
            dryRun = true;
        }
        else if (!gotDupCount)
        {
            if (!parseU64(strArg, &dupCount))
            {
                usage();
                return 1;
            }
            gotDupCount = true;
        }
        else if (!gotUniqueCount)
        {
            if (!parseU64(strArg, &uniqueCount))
            {
                usage();
                return 1;
            }
            gotUniqueCount = true;
        }
        else
        {
            usage();
            return 1;
        }
    }

    if (help)
    {
        usage();
        return 0;
    }

    if (!gotDupCount || !gotUniqueCount)
    {
        usage();
        return 1;
    }

    Solver solver(dryRun, dupCount, uniqueCount);

    return solver.solve();
}
