#include "Common.hpp"
#include "ParseUtils.hpp"
#include "Cleanup.hpp"
#include "Protocol.hpp"
#include "Solution.hpp"
#include "Xoshiro.hpp"
#include "FastRandom.hpp"
#include "TimeUtils.hpp"
#include "ProcessId.hpp"
#include <array>

using std::string;
using std::vector;
using std::array;

typedef Xoshiro256StarStar Gen;

class PlanMove
{
public:
    PlanMove() : planIdx(0), moveIdx(0) {}
    PlanMove(uint64_t planIdx_, uint64_t moveIdx_) : planIdx(planIdx_), moveIdx(moveIdx_) {}

    bool operator==(const PlanMove& other) const
    {
        return planIdx == other.planIdx && moveIdx == other.moveIdx;
    }

    bool operator!=(const PlanMove& other) const
    {
        return planIdx != other.planIdx || moveIdx != other.moveIdx;
    }

    uint64_t planIdx = 0;
    uint64_t moveIdx = 0;
};

void usage()
{
    printf("Usage: lightning2 <size>\n");
    printf("  <size>\n");
    printf("        Problem size\n");
    printf("Options:\n");
    printf("  -h    Print usage information and exit\n");
    printf("  -d    Dry run: don't actually submit\n");
}

int main(int argc, char *argv[])
{
    bool gotSize = false;

    bool help = false;
    bool dryRun = false;
    uint64_t size = 0;

    int iArg = 1;
    while (iArg < argc)
    {
        string strArg = argv[iArg++];

        if (strArg == "-h" || strArg == "--help")
        {
            help = true;
        }
        else if (strArg == "-d")
        {
            dryRun = true;
        }
        else if (!gotSize)
        {
            if (!parseU64(strArg, &size))
            {
                usage();
                return 1;
            }
            gotSize = true;
        }
        else
        {
            usage();
            return 1;
        }
    }

    if (help)
    {
        usage();
        return 0;
    }

    if (!gotSize)
    {
        usage();
        return 1;
    }

    string problemName;

    switch (size)
    {
    case 3: problemName = "probatio"; break;
    case 6: problemName = "primus"; break;
    case 12: problemName = "secundus"; break;
    case 18: problemName = "tertius"; break;
    case 24: problemName = "quartus"; break;
    case 30: problemName = "quintus"; break;
    default:
        printf("Unknown problem size\n");
        return 1;
    }

    printf("problem: %s\n", problemName.c_str());
    printf("size: %" PRIu64 "\n", size);

    Protocol::init();
    Cleanup cleanupProtocol([](){ Protocol::cleanup(); });

    string msg;

    if (!Protocol::select(problemName, &msg))
    {
        printf("%s\n", msg.c_str());
        return 1;
    }

    Gen gen;
#if 1
    {
        uint64_t timeMS = getTimeMS();
        uint32_t pid = getProcessId();
        uint32_t seed0 = 0xddf294ea;
        uint32_t seed1 = (uint32_t)timeMS;
        uint32_t seed2 = (uint32_t)pid ^ (uint32_t)(timeMS >> 32);
        std::seed_seq seq{seed0, seed1, seed2};
        gen.seed(seq);
    }
#endif
#if 0
    {
        uint32_t seed0 = 0xddf294ea;
        std::seed_seq seq{seed0};
        gen.seed(seq);
    }
#endif

    uint64_t planCount = 1;
    uint64_t moveCount = size * 18;

    vector<string> plans(planCount);
    for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
    {
        auto& plan = plans[planIdx];
        plan.resize(moveCount);
        for (uint64_t moveIdx = 0; moveIdx < moveCount; moveIdx++)
        {
            plan[moveIdx] = '0' + fastRandom((uint32_t)6, gen);
        }
    }

    for (auto& plan : plans)
    {
        printf("%s\n", plan.c_str());
    }

    vector<vector<uint8_t>> results;
    uint64_t queryCount;
    if (!Protocol::explore(plans, &results, &queryCount, &msg))
    {
        printf("%s\n", msg.c_str());
        return 1;
    }

    for (auto& result : results)
    {
        for (uint8_t value : result)
        {
            printf("%" PRIu8 "", value);
        }
        printf("\n");
    }
    printf("queryCount = %" PRIu64 "\n", queryCount);

    while (true)
    {
        vector<vector<uint64_t>> roomMap(planCount);
        for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
        {
            roomMap[planIdx].resize(moveCount + 1);
            roomMap[planIdx][0] = results[planIdx][0];
            for (uint64_t moveIdx = 0; moveIdx < moveCount; moveIdx++)
            {
                uint64_t value = results[planIdx][moveIdx + 1];
                uint64_t valueCount = (size + 3 - value) / 4;
                roomMap[planIdx][moveIdx + 1] = fastRandom(valueCount, gen) * 4 + value;
            }
        }

        vector<array<RoomDoor, 6>> connections(size);
        for (uint64_t roomIdx = 0; roomIdx < size; roomIdx++)
        {
            for (uint64_t doorIdx = 0; doorIdx < 6; doorIdx++)
            {
                auto& roomDoor = connections[roomIdx][doorIdx];
                roomDoor.room = roomIdx;
                roomDoor.door = doorIdx;
            }
        }

        for (uint64_t i = 0; i < size * 6; i++)
        {
            uint64_t fromRoomIdx = fastRandom(size, gen);
            uint64_t fromDoorIdx = fastRandom(6, gen);
            auto& fromRoomDoor = connections[fromRoomIdx][fromDoorIdx];
            if (fromRoomDoor.room != fromRoomIdx ||
                fromRoomDoor.door != fromDoorIdx)
            {
                continue;
            }

            uint64_t toRoomIdx = fastRandom(size, gen);
            uint64_t toDoorIdx = fastRandom(6, gen);
            if (toRoomIdx == fromRoomIdx && toDoorIdx == fromDoorIdx)
            {
                continue;
            }
            auto& toRoomDoor = connections[toRoomIdx][toDoorIdx];
            if (toRoomDoor.room != toRoomIdx ||
                toRoomDoor.door != toDoorIdx)
            {
                continue;
            }

            fromRoomDoor.room = toRoomIdx;
            fromRoomDoor.door = toDoorIdx;
            toRoomDoor.room = fromRoomIdx;
            toRoomDoor.door = fromDoorIdx;
        }

        vector<array<vector<PlanMove>, 6>> doorMap(size);
        for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
        {
            for (uint64_t moveIdx = 0; moveIdx < moveCount; moveIdx++)
            {
                uint64_t roomIdx = roomMap[planIdx][moveIdx];
                uint64_t doorIdx = plans[planIdx][moveIdx] - '0';
                doorMap[roomIdx][doorIdx].emplace_back(planIdx, moveIdx);
            }
        }

        auto calcCost = [&]() -> uint64_t
        {
            uint64_t cost = 0;
            for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
            {
                for (uint64_t moveIdx = 0; moveIdx < moveCount; moveIdx++)
                {
                    uint64_t fromRoomIdx = roomMap[planIdx][moveIdx];
                    uint64_t toRoomIdx = roomMap[planIdx][moveIdx + 1];
                    uint64_t doorIdx = plans[planIdx][moveIdx] - '0';
                    if (connections[fromRoomIdx][doorIdx].room != toRoomIdx)
                    {
                        cost++;
                    }
                }
            }
            return cost;
        };

        uint64_t cost = calcCost();
        uint64_t bestCost = cost;
        printf("%9" PRIu64 "  %4" PRIu64 "\n", (uint64_t)0, cost);

        double allowedLossInitial = 0.3;
        double allowedLossFinal = 0.2;
        double lnRatio = std::log(allowedLossFinal / allowedLossInitial);

        double allowedLoss = allowedLossInitial;
        double recipAllowedLoss = 1.0 / allowedLoss;

        vector<Connection> undoLog;

        uint64_t iterLimit = size * size * 1000000;

        uint64_t iter = 0;
        while (iter < iterLimit)
        {
            iter++;

            if ((iter & 0xf) == 0)
            {
                double param = (double)iter / (double)iterLimit;
                allowedLoss = allowedLossInitial * std::exp(lnRatio * param);
                recipAllowedLoss = 1.0 / allowedLoss;
            }

            uint64_t changeType = fastRandom((uint64_t)2, gen);
            uint64_t planIdx = (uint64_t)-1;
            uint64_t moveIdx = (uint64_t)-1;
            uint64_t oldRoomIdx = (uint64_t)-1;
            uint64_t newRoomIdx = (uint64_t)-1;

            uint64_t curCost = cost;

            if (changeType == 0)
            {
                planIdx = fastRandom(planCount, gen);
                moveIdx = fastRandom(moveCount, gen);
                oldRoomIdx = roomMap[planIdx][moveIdx + 1];
                uint64_t value = results[planIdx][moveIdx + 1];
                uint64_t valueCount = (size + 3 - value) / 4;
                newRoomIdx = fastRandom(valueCount, gen) * 4 + value;
                if (newRoomIdx == oldRoomIdx) continue;
                roomMap[planIdx][moveIdx + 1] = newRoomIdx;

                {
                    uint64_t prevRoomIdx = roomMap[planIdx][moveIdx];
                    uint64_t doorIdx = plans[planIdx][moveIdx] - '0';
                    if (connections[prevRoomIdx][doorIdx].room != oldRoomIdx)
                    {
                        curCost--;
                    }
                    if (connections[prevRoomIdx][doorIdx].room != newRoomIdx)
                    {
                        curCost++;
                    }
                }

                if (moveIdx + 1 < moveCount)
                {
                    uint64_t nextRoomIdx = roomMap[planIdx][moveIdx + 2];
                    uint64_t doorIdx = plans[planIdx][moveIdx + 1] - '0';
                    if (connections[oldRoomIdx][doorIdx].room != nextRoomIdx)
                    {
                        curCost--;
                    }
                    if (connections[newRoomIdx][doorIdx].room != nextRoomIdx)
                    {
                        curCost++;
                    }
                }

                //curCost = calcCost();
            }
            else if (changeType == 1)
            {
                uint64_t fromRoomIdx = fastRandom(size, gen);
                uint64_t fromDoorIdx = fastRandom(6, gen);
                auto fromRoomDoor = connections[fromRoomIdx][fromDoorIdx];

                uint64_t toRoomIdx = fastRandom(size, gen);
                uint64_t toDoorIdx = fastRandom(6, gen);
                auto toRoomDoor = connections[toRoomIdx][toDoorIdx];

                if (toRoomIdx == fromRoomIdx &&
                    toDoorIdx == fromDoorIdx)
                {
                    continue;
                }

                if (fromRoomDoor.room == toRoomIdx &&
                    fromRoomDoor.door == toDoorIdx)
                {
                    continue;
                }

                undoLog.clear();

                {
                    auto& conn = undoLog.emplace_back();
                    conn.from.room = fromRoomIdx;
                    conn.from.door = fromDoorIdx;
                    conn.to = fromRoomDoor;
                }

                {
                    auto& conn = undoLog.emplace_back();
                    conn.from.room = toRoomIdx;
                    conn.from.door = toDoorIdx;
                    conn.to = toRoomDoor;
                }

                auto updateCost = [&](uint64_t fromRoomIdx,
                                      uint64_t fromDoorIdx,
                                      uint64_t oldToRoomIdx,
                                      uint64_t newToRoomIdx)
                {
                    for (auto& planMove : doorMap[fromRoomIdx][fromDoorIdx])
                    {
                        if (oldToRoomIdx != roomMap[planMove.planIdx][planMove.moveIdx + 1])
                        {
                            curCost--;
                        }
                        if (newToRoomIdx != roomMap[planMove.planIdx][planMove.moveIdx + 1])
                        {
                            curCost++;
                        }
                    }
                };

                if ((fromRoomDoor.room != fromRoomIdx ||
                     fromRoomDoor.door != fromDoorIdx) &&
                    (toRoomDoor.room != toRoomIdx ||
                     toRoomDoor.door != toDoorIdx))
                {
                    connections[fromRoomDoor.room][fromRoomDoor.door] = toRoomDoor;
                    connections[toRoomDoor.room][toRoomDoor.door] = fromRoomDoor;

                    updateCost(fromRoomDoor.room, fromRoomDoor.door, fromRoomIdx, toRoomDoor.room);
                    updateCost(toRoomDoor.room, toRoomDoor.door, toRoomIdx, fromRoomDoor.room);
                }
                else
                {
                    if (fromRoomDoor.room != fromRoomIdx ||
                        fromRoomDoor.door != fromDoorIdx)
                    {
                        connections[fromRoomDoor.room][fromRoomDoor.door] = fromRoomDoor;
                        updateCost(fromRoomDoor.room, fromRoomDoor.door, fromRoomIdx, fromRoomDoor.room);
                    }

                    if (toRoomDoor.room != toRoomIdx ||
                        toRoomDoor.door != toDoorIdx)
                    {
                        connections[toRoomDoor.room][toRoomDoor.door] = toRoomDoor;
                        updateCost(toRoomDoor.room, toRoomDoor.door, toRoomIdx, toRoomDoor.room);
                    }
                }

                connections[fromRoomIdx][fromDoorIdx].room = toRoomIdx;
                connections[fromRoomIdx][fromDoorIdx].door = toDoorIdx;
                connections[toRoomIdx][toDoorIdx].room = fromRoomIdx;
                connections[toRoomIdx][toDoorIdx].door = fromDoorIdx;

                updateCost(fromRoomIdx, fromDoorIdx, fromRoomDoor.room, toRoomIdx);
                updateCost(toRoomIdx, toDoorIdx, toRoomDoor.room, fromRoomIdx);

                //curCost = calcCost();
            }

#if 0
            uint64_t slowCost = calcCost();
            if (slowCost != curCost)
            {
                printf("Incremental cost error!\n");
                return 1;
            }
#endif

            bool keep;

            if (curCost <= cost)
            {
                keep = true;
            }
            else
            {
                double loss = (double)(curCost - cost);
#if 0
                keep = false;
#endif
#if 1
                keep = fastRandomDouble(gen) < std::exp(-loss * recipAllowedLoss);
#endif
#if 0
                keep = fastRandomDouble(gen) >= loss * recipAllowedLoss;
#endif
            }

            if (keep)
            {
                cost = curCost;
                if (cost < bestCost)
                {
                    printf("%9" PRIu64 "  %4" PRIu64 "\n", iter, cost);
                    bestCost = cost;
                }

                if (changeType == 0)
                {
                    if (moveIdx + 1 < moveCount)
                    {
                        uint64_t doorIdx = plans[planIdx][moveIdx + 1] - '0';
                        std::erase(doorMap[oldRoomIdx][doorIdx], PlanMove(planIdx, moveIdx + 1));
                        doorMap[newRoomIdx][doorIdx].emplace_back(planIdx, moveIdx + 1);
                    }

#if 0
                    for (uint64_t roomIdx = 0; roomIdx < size; roomIdx++)
                    {
                        for (uint64_t doorIdx = 0; doorIdx < 6; doorIdx++)
                        {
                            for (auto& planMove : doorMap[roomIdx][doorIdx])
                            {
                                if (roomMap[planMove.planIdx][planMove.moveIdx] != roomIdx ||
                                    (uint64_t)(plans[planMove.planIdx][planMove.moveIdx] - '0') != doorIdx)
                                {
                                    printf("doorMap error! (1)\n");
                                    return 1;
                                }
                            }
                        }
                    }

                    for (uint64_t planIdx = 0; planIdx < planCount; planIdx++)
                    {
                        for (uint64_t moveIdx = 0; moveIdx < moveCount; moveIdx++)
                        {
                            uint64_t roomIdx = roomMap[planIdx][moveIdx];
                            uint64_t doorIdx = plans[planIdx][moveIdx] - '0';
                            if (std::find(doorMap[roomIdx][doorIdx].begin(),
                                          doorMap[roomIdx][doorIdx].end(),
                                          PlanMove(planIdx, moveIdx)) ==
                                doorMap[roomIdx][doorIdx].end())
                            {
                                printf("doorMap error! (2)\n");
                                return 1;
                            }
                        }
                    }
#endif
                }
            }
            else
            {
                if (changeType == 0)
                {
                    roomMap[planIdx][moveIdx + 1] = oldRoomIdx;
                }
                else if (changeType == 1)
                {
                    for (auto& conn : undoLog)
                    {
                        connections[conn.from.room][conn.from.door] = conn.to;
                        connections[conn.to.room][conn.to.door] = conn.from;
                    }
                }
            }

            if (bestCost == 0) break;
        }

        if (bestCost != 0) continue;

        Solution solution;
        for (uint64_t roomIdx = 0; roomIdx < size; roomIdx++)
        {
            solution.rooms.push_back(roomIdx % 4);
        }

        solution.startingRoom = results[0][0];

        for (uint64_t fromRoomIdx = 0; fromRoomIdx < size; fromRoomIdx++)
        {
            for (uint64_t fromDoorIdx = 0; fromDoorIdx < 6; fromDoorIdx++)
            {
                auto& fromRoomDoor = connections[fromRoomIdx][fromDoorIdx];
                uint64_t toRoomIdx = fromRoomDoor.room;
                uint64_t toDoorIdx = fromRoomDoor.door;

                if ((fromRoomIdx < toRoomIdx) ||
                    (fromRoomIdx == toRoomIdx && fromDoorIdx <= toDoorIdx))
                {
                    auto& connection = solution.connections.emplace_back();
                    connection.from.room = fromRoomIdx;
                    connection.from.door = fromDoorIdx;
                    connection.to.room = toRoomIdx;
                    connection.to.door = toDoorIdx;
                }
            }
        }

        if (dryRun)
        {
            printf("Dry run: not submitting\n");
        }
        else
        {
            bool correct;
            if (!Protocol::guess(solution, &correct, &msg))
            {
                printf("%s\n", msg.c_str());
                return 1;
            }

            printf("correct = %s\n", correct ? "true" : "false");
        }

        break;
    }

    return 0;
}
