#include "Common.hpp"
#include "Cleanup.hpp"
#include "Protocol.hpp"
#include "Solution.hpp"

using std::string;
using std::vector;

void usage(){

    printf("Usage:\n");
    printf("       manual select <problem>\n");
    printf("       manual explore <plan>\n");
    printf("       manual guess\n");
    printf("  <problem>\n");
    printf("        Problem name\n");
    printf("  <plan>\n");
    printf("        Exploration plan\n");
    printf("Options:\n");
    printf("  -h    Print usage information and exit\n");
}

int main(int argc, char *argv[])
{
    bool gotCommand = false;
    bool gotProblemName = false;
    bool gotPlan = false;

    bool help = false;
    bool selectCommand = false;
    bool exploreCommand = false;
    bool guessCommand = false;
    string problemName;
    string plan;

    int iArg = 1;
    while (iArg < argc)
    {
        string strArg = argv[iArg++];

        if (strArg == "-h" || strArg == "--help")
        {
            help = true;
        }
        else if (!gotCommand)
        {
            if (strArg == "select") selectCommand = true;
            else if (strArg == "explore") exploreCommand = true;
            else if (strArg == "guess") guessCommand = true;
            else
            {
                usage();
                return 1;
            }
            gotCommand = true;
        }
        else if (selectCommand && !gotProblemName)
        {
            problemName = strArg;
            gotProblemName = true;
        }
        else if (exploreCommand && !gotPlan)
        {
            plan = strArg;
            gotPlan = true;
        }
        else
        {
            usage();
            return 1;
        }
    }

    if (help)
    {
        usage();
        return 0;
    }

    if (!gotCommand ||
        (selectCommand && !gotProblemName) ||
        (exploreCommand && !gotPlan))
    {
        usage();
        return 1;
    }

    Protocol::init();
    Cleanup cleanupProtocol([](){ Protocol::cleanup(); });

    string msg;

    if (selectCommand)
    {
        if (!Protocol::select(problemName, &msg))
        {
            printf("%s\n", msg.c_str());
            return 1;
        }
    }
    else if (exploreCommand)
    {
        vector<string> plans;
        plans.push_back(plan);

        vector<vector<uint8_t>> results;
        uint64_t queryCount;
        if (!Protocol::explore(plans, &results, &queryCount, &msg))
        {
            printf("%s\n", msg.c_str());
            return 1;
        }

        for (auto& result : results)
        {
            for (uint8_t value : result)
            {
                printf("%" PRIu8 "", value);
            }
            printf("\n");
        }
        printf("queryCount = %" PRIu64 "\n", queryCount);
    }
    else if (guessCommand)
    {
        Solution solution;

        uint64_t dupCount = 2;
        uint64_t uniqueCount = 6;
        uint64_t roomCount = dupCount * uniqueCount;

        for (uint64_t dupIdx = 0; dupIdx < dupCount; dupIdx++)
        {
            for (uint64_t uniqueIdx = 0; uniqueIdx < uniqueCount; uniqueIdx++)
            {
                solution.rooms.push_back(uniqueIdx % 4);
            }
        }

        for (uint64_t roomIdx = 0; roomIdx < roomCount; roomIdx++)
        {
            uint64_t value = solution.rooms[roomIdx];
            printf("%" PRIu64 ": %" PRIu64 "\n", roomIdx, value);
        }
        
        solution.startingRoom = 0;

        printf("start: %" PRIu64 "\n", (uint64_t)solution.startingRoom);

        uint64_t A = 0;
        uint64_t B = 1;
        uint64_t C = 2;
        uint64_t D = 3;
        uint64_t E = 4;
        uint64_t F = 5;
        //uint64_t G = 6;
        //uint64_t H = 7;
        //uint64_t I = 8;
        uint64_t J = 9;
        //uint64_t K = 10;
        uint64_t L = 11;

        auto add = [&](uint64_t roomIdxI, uint64_t doorIdxI, uint64_t roomIdxJ, uint64_t doorIdxJ)
        {
            uint64_t dupIdxI = roomIdxI / uniqueCount;
            uint64_t uniqueIdxI = roomIdxI % uniqueCount;
            uint64_t dupIdxJ = roomIdxJ / uniqueCount;
            uint64_t uniqueIdxJ = roomIdxJ % uniqueCount;

            {
                auto& connection = solution.connections.emplace_back();
                connection.from.room = roomIdxI;
                connection.from.door = doorIdxI;
                connection.to.room = roomIdxJ;
                connection.to.door = doorIdxJ;
            }

            {
                auto& connection = solution.connections.emplace_back();
                connection.from.room = (1 - dupIdxI) * uniqueCount + uniqueIdxI;
                connection.from.door = doorIdxI;
                connection.to.room = (1 - dupIdxJ) * uniqueCount + uniqueIdxJ;
                connection.to.door = doorIdxJ;
            }
        };

        add(A, 0, F, 3);
        add(A, 1, A, 1);
        add(A, 2, B, 4);
        add(A, 3, A, 3);
        add(A, 4, B, 5);
        add(A, 5, E, 3);

        add(B, 0, B, 0);
        add(B, 1, C, 5);
        add(B, 2, L, 4);
        add(B, 3, D, 4);

        add(C, 0, E, 4);
        add(C, 1, E, 5);
        add(C, 2, D, 2);
        add(C, 3, J, 0);
        add(C, 4, J, 1);

        add(D, 3, F, 2);
        add(D, 5, F, 5);

        add(E, 0, E, 0);
        add(E, 1, E, 1);
        add(E, 2, F, 0);

        add(F, 1, F, 1);

        for (auto& connection : solution.connections)
        {
            printf("(%" PRIu64 ", %" PRIu64 ") - (%" PRIu64 ", %" PRIu64 ")\n",
                   connection.from.room,
                   connection.from.door,
                   connection.to.room,
                   connection.to.door);
        }

#if 0
        bool correct;
        if (!Protocol::guess(solution, &correct, &msg))
        {
            printf("%s\n", msg.c_str());
            return 1;
        }

        printf("correct = %s\n", correct ? "true" : "false");
#endif
    }

    return 0;
}
